%% Fitting a transition model to the data: fossil enamel stable C isotopes
%
% by Martin H. Trauth, 24 August 2020
% http://mres.uni-potsdam.de

%%
% Bibi, F., Souron, A., Bocherens, H., Uno, K., Boisserie, J. (2012).
% Ecological change in the lower Omo Valley around 2.8 Ma Biology Letters
% 9(1), 20120890-20120890. https://dx.doi.org/10.1098/rsbl.2012.0890
clear, clc, close all
dataraw = datastore('data_bibi_2012.txt');
data = readall(dataraw);
data.Age = -data.Age;

%%
fo = fitoptions('Method','NonlinearLeastSquares',...
    'StartPoint',[0 0 -2.8 1],...
    'Lower',[-20 -10 -5  0],...
    'Upper',[ 20  10  5 10]);
ft = fittype(@(a,b,c,d,x) a+b*(1./(1+exp(-d*(x-c)))),...
    'Coeff',{'a','b','c','d'},...
    'Options',fo);

%%
t = -3.5 : 0.1 : -1;
[f,fgof,fout] = fit(data.Age,data.d13C,ft)
y = f.a+f.b*(1./(1+exp(-f.d*(t-f.c))));

%%
rft = load('rampfit.dat');
rs = 'c = -2.851 (-2.945,-2.757)';

%%
xl = ['Age (Ma)'];
yl = ['\delta^{13}C_{enamel} (',char(8240),')'];

%%
close all
figure('Position',[50 1000 800 250])
axes('Position',[0.1 0.2 0.8 0.7],...
    'XLim',[-4 -1],...
    'YLim',[-10 2],...
    'FontSize',12,...
    'Box','On'), hold on
p = plot(f,data.Age,data.d13C);            % Sigmoid Fit and Data
p(1).Color = [0.1 0.3 0.6];
p(1).Marker = 'Square';
p(1).MarkerSize = 6;
p(1).MarkerFaceColor = [0.1 0.3 0.6];
p(1).MarkerEdgeColor = [0.1 0.3 0.6];
p(2).Color = [0.7 0.3 0.1];
p(2).LineWidth = 1.5;
p(2).LineStyle = '-';

line(-rft(:,1),rft(:,3),...                % Ramp Fit
    'Color',[0.7 0.3 0.1],...
    'LineWidth',1.5,...
    'LineStyle','-.')

lgd = legend('Data Points','Sigmoid Fit','Ramp Fit');
lgd.Location = 'southeast';
lgd.Box = 'Off';
lgd.FontSize = 12;
lgd.AutoUpdate = 'Off';
xlabel(xl,...
    'FontSize',12)
ylabel(yl,...
    'FontSize',12)

line([-2.85 -2.85],[-4 -2],...              % Inflection Sigmoid Fit
    'LineWidth',1.5,...
    'Color',[0.7 0.3 0.1])
text(-2.85,-1,'-2.85 Ma',...
    'FontSize',18,...
    'Color',[0.7 0.3 0.1],...
    'HorizontalAlignment','Right')

line([-2.85 -2.85],[-8 -6],...              % Inflection Ramp Fit
    'LineWidth',1.5,...
    'LineStyle','-.',...
    'Color',[0.7 0.3 0.1])
text(-2.85,-9,'-2.85 Ma',...
    'FontSize',18,...
    'Color',[0.7 0.3 0.1],...
    'HorizontalAlignment','Left')

print -dpng -r300 commands_bibi2012_sigmoid_ramp_vs2.png

